<?php
/**
 * Plugin Name: Price Comparison Tables
 * Plugin URI: https://wordpress.org/plugins/price-comparison-tables
 * Description: Create beautiful, responsive price comparison tables with an intuitive admin interface. Perfect for affiliate sites, SaaS comparisons, and product reviews.
 * Version: 1.0.0
 * Author: Buzzettycom
 * Author URI: https://buzzetty.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: price-comparison-tables
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Main Price Comparison Tables Class
 */
class Price_Comparison_Tables
{

    /**
     * Plugin version
     */
    const VERSION = '1.0.0';

    /**
     * Post type name
     */
    const POST_TYPE = 'pct_table';

    /**
     * Constructor
     */
    public function __construct()
    {
        // Register custom post type
        add_action('init', array($this, 'register_post_type'));

        // Add meta boxes
        add_action('add_meta_boxes', array($this, 'add_meta_boxes'));

        // Save post meta
        add_action('save_post', array($this, 'save_table_data'));

        // Register shortcode
        add_shortcode('comparison_table', array($this, 'render_table'));

        // Enqueue admin assets
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));

        // Enqueue frontend assets
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'));

        // Add settings link
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($this, 'add_settings_link'));

        // Add columns to admin list
        add_filter('manage_' . self::POST_TYPE . '_posts_columns', array($this, 'add_admin_columns'));
        add_action('manage_' . self::POST_TYPE . '_posts_custom_column', array($this, 'render_admin_columns'), 10, 2);
    }

    /**
     * Register custom post type
     */
    public function register_post_type()
    {
        $labels = array(
            'name' => 'Comparison Tables',
            'singular_name' => 'Comparison Table',
            'menu_name' => 'Price Tables',
            'add_new' => 'Add New',
            'add_new_item' => 'Add New Table',
            'edit_item' => 'Edit Table',
            'new_item' => 'New Table',
            'view_item' => 'View Table',
            'search_items' => 'Search Tables',
            'not_found' => 'No tables found',
            'not_found_in_trash' => 'No tables found in trash'
        );

        $args = array(
            'labels' => $labels,
            'public' => false,
            'show_ui' => true,
            'show_in_menu' => true,
            'menu_icon' => 'dashicons-editor-table',
            'capability_type' => 'post',
            'hierarchical' => false,
            'supports' => array('title'),
            'has_archive' => false,
            'rewrite' => false,
            'query_var' => false,
        );

        register_post_type(self::POST_TYPE, $args);
    }

    /**
     * Add meta boxes
     */
    public function add_meta_boxes()
    {
        add_meta_box(
            'pct_table_builder',
            'Table Builder',
            array($this, 'render_table_builder'),
            self::POST_TYPE,
            'normal',
            'high'
        );

        add_meta_box(
            'pct_shortcode',
            'Shortcode',
            array($this, 'render_shortcode_box'),
            self::POST_TYPE,
            'side',
            'default'
        );
    }

    /**
     * Render table builder meta box
     */
    public function render_table_builder($post)
    {
        wp_nonce_field('pct_save_table', 'pct_nonce');

        // Get existing data
        $table_data = get_post_meta($post->ID, '_pct_table_data', true);
        if (empty($table_data)) {
            $table_data = $this->get_default_table_data();
        } else {
            $table_data = json_decode($table_data, true);
        }

        ?>
        <div id="pct-builder-app">
            <div class="pct-builder-container">

                <!-- Products Section -->
                <div class="pct-section">
                    <h3>Products / Plans</h3>
                    <button type="button" class="button pct-add-product">+ Add Product</button>

                    <div class="pct-products-list" id="pct-products">
                        <!-- Products will be rendered here by JavaScript -->
                    </div>
                </div>

                <!-- Features Section -->
                <div class="pct-section">
                    <h3>Features / Rows</h3>
                    <button type="button" class="button pct-add-feature">+ Add Feature</button>

                    <div class="pct-features-list" id="pct-features">
                        <!-- Features will be rendered here by JavaScript -->
                    </div>
                </div>

                <!-- Styling Section -->
                <div class="pct-section">
                    <h3>Styling</h3>
                    <table class="form-table">
                        <tr>
                            <th><label for="pct-theme">Theme</label></th>
                            <td>
                                <select id="pct-theme">
                                    <?php
                                    $themes = $this->get_available_themes();
                                    $current_theme = isset($table_data['settings']['theme']) ? $table_data['settings']['theme'] : 'simple';
                                    foreach ($themes as $key => $label): ?>
                                        <option value="<?php echo esc_attr($key); ?>" <?php selected($current_theme, $key); ?>>
                                            <?php echo esc_html($label); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </td>
                        </tr>
                        <tr>
                            <th><label for="pct-accent-color">Accent Color</label></th>
                            <td>
                                <input type="color" id="pct-accent-color"
                                    value="<?php echo esc_attr($table_data['settings']['accent_color']); ?>">
                            </td>
                        </tr>
                        <tr>
                            <th><label for="pct-highlight-color">Highlight Color</label></th>
                            <td>
                                <input type="color" id="pct-highlight-color"
                                    value="<?php echo esc_attr($table_data['settings']['highlight_color']); ?>">
                            </td>
                        </tr>
                    </table>
                </div>

                <!-- Hidden field to store JSON data -->
                <input type="hidden" id="pct-table-data" name="pct_table_data" value="">
            </div>
        </div>

        <script type="text/javascript">
            // Pass initial data to JavaScript
            var pctInitialData = <?php echo json_encode($table_data); ?>;
        </script>
        <?php
    }

    /**
     * Render shortcode meta box
     */
    public function render_shortcode_box($post)
    {
        if ($post->post_status === 'publish') {
            ?>
            <p>Use this shortcode to display the table:</p>
            <input type="text" readonly value='[comparison_table id="<?php echo esc_attr($post->ID); ?>"]' onclick="this.select();"
                style="width: 100%; padding: 8px; font-family: monospace;">
            <p class="description">Click to select, then copy and paste into your post or page.</p>
            <?php
        } else {
            ?>
            <p>Publish this table to get the shortcode.</p>
            <?php
        }
    }

    /**
     * Get default table data
     */
    private function get_default_table_data()
    {
        return array(
            'products' => array(
                array(
                    'id' => 'prod_1',
                    'name' => 'Basic',
                    'price' => '$9/mo',
                    'badge' => '',
                    'highlighted' => false,
                    'button_text' => 'Get Started',
                    'button_link' => '#',
                    'features' => array()
                ),
                array(
                    'id' => 'prod_2',
                    'name' => 'Pro',
                    'price' => '$29/mo',
                    'badge' => 'Popular',
                    'highlighted' => true,
                    'button_text' => 'Get Started',
                    'button_link' => '#',
                    'features' => array()
                ),
                array(
                    'id' => 'prod_3',
                    'name' => 'Enterprise',
                    'price' => '$99/mo',
                    'badge' => '',
                    'highlighted' => false,
                    'button_text' => 'Contact Us',
                    'button_link' => '#',
                    'features' => array()
                )
            ),
            'features' => array(
                array('id' => 'feat_1', 'name' => 'Storage'),
                array('id' => 'feat_2', 'name' => 'Users'),
                array('id' => 'feat_3', 'name' => 'Support')
            ),
            'settings' => array(
                'accent_color' => '#007bff',
                'highlight_color' => '#28a745'
            )
        );
    }

    /**
     * Save table data
     */
    public function save_table_data($post_id)
    {
        // Check nonce
        if (!isset($_POST['pct_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['pct_nonce'])), 'pct_save_table')) {
            return;
        }

        // Check autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        // Check permissions
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        // Check post type
        if (get_post_type($post_id) !== self::POST_TYPE) {
            return;
        }

        // Save table data
        if (isset($_POST['pct_table_data'])) {
            // Data is sent as Base64 encoded JSON to avoid sanitization issues
            $encoded_data = sanitize_text_field(wp_unslash($_POST['pct_table_data']));

            // Allow empty data, but if set, try to decode
            if (empty($encoded_data)) {
                update_post_meta($post_id, '_pct_table_data', '');
                return;
            }

            // 1. Decode Base64
            $json_data = base64_decode($encoded_data);

            // 2. Validate it's proper JSON
            if ($json_data !== false && json_decode($json_data) !== null) {
                // 3. Save the raw JSON
                // We use update_post_meta directly as we want to store the exact JSON structure
                update_post_meta($post_id, '_pct_table_data', $json_data);
            }
        }
    }

    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook)
    {
        global $post_type;

        if (($hook === 'post.php' || $hook === 'post-new.php') && $post_type === self::POST_TYPE) {
            wp_enqueue_style(
                'pct-admin-style',
                plugin_dir_url(__FILE__) . 'assets/css/admin.css',
                array(),
                self::VERSION
            );

            wp_enqueue_script(
                'pct-admin-script',
                plugin_dir_url(__FILE__) . 'assets/js/admin-builder.js',
                array('jquery'),
                self::VERSION,
                true
            );
        }
    }

    /**
     * Enqueue frontend assets
     */
    public function enqueue_frontend_assets()
    {
        wp_register_style(
            'pct-frontend-style',
            plugin_dir_url(__FILE__) . 'assets/css/frontend.css',
            array(),
            self::VERSION
        );

        wp_register_script(
            'pct-frontend-script',
            plugin_dir_url(__FILE__) . 'assets/js/frontend.js',
            array(),
            self::VERSION,
            true
        );
    }

    /**
     * Render comparison table
     */
    public function render_table($atts)
    {
        $atts = shortcode_atts(
            array(
                'id' => 0,
            ),
            $atts,
            'comparison_table'
        );

        $table_id = intval($atts['id']);

        if ($table_id === 0) {
            return '<p>Please specify a table ID.</p>';
        }

        // Get table data
        $table_data = get_post_meta($table_id, '_pct_table_data', true);

        if (empty($table_data)) {
            return '<p>Table not found.</p>';
        }

        $table_data = json_decode($table_data, true);

        // Enqueue assets
        wp_enqueue_style('pct-google-fonts', 'https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700;800&display=swap', array(), self::VERSION);
        wp_enqueue_style('pct-frontend-style');
        wp_enqueue_script('pct-frontend-script');

        // Render table
        ob_start();
        include plugin_dir_path(__FILE__) . 'templates/table-template.php';
        return ob_get_clean();
    }

    /**
     * Add settings link
     */
    public function add_settings_link($links)
    {
        $settings_link = '<a href="edit.php?post_type=' . self::POST_TYPE . '">Tables</a>';
        array_unshift($links, $settings_link);
        return $links;
    }

    /**
     * Add admin columns
     */
    public function add_admin_columns($columns)
    {
        $new_columns = array();
        $new_columns['cb'] = $columns['cb'];
        $new_columns['title'] = $columns['title'];
        $new_columns['shortcode'] = 'Shortcode';
        $new_columns['products'] = 'Products';
        $new_columns['date'] = $columns['date'];
        return $new_columns;
    }

    /**
     * Render admin columns
     */
    public function render_admin_columns($column, $post_id)
    {
        switch ($column) {
            case 'shortcode':
                echo '<code>[comparison_table id="' . esc_attr($post_id) . '"]</code>';
                break;
            case 'products':
                $table_data = get_post_meta($post_id, '_pct_table_data', true);
                if (!empty($table_data)) {
                    $data = json_decode($table_data, true);
                    echo count($data['products']) . ' products';
                }
                break;
        }
    }

    /**
     * Get available themes
     */
    public function get_available_themes()
    {
        $themes = array(
            'simple' => 'Simple (Default)',
            'professional' => 'Professional',
            'modern' => 'Modern'
        );
        return apply_filters('pct_available_themes', $themes);
    }
}

// Initialize the plugin
new Price_Comparison_Tables();

