/**
 * Price Comparison Tables - Matrix Builder
 */

(function ($) {
    'use strict';

    let tableData = {};

    /**
     * Initialize builder
     */
    $(document).ready(function () {
        if (typeof pctInitialData !== 'undefined') {
            tableData = pctInitialData;

            // Seed default data if empty (New Table)
            if (!tableData.products || tableData.products.length === 0) {
                seedDefaultData();
            }

            renderBuilder();
        }

        // Global Listeners (Delegated)
        // const container = $('#pct-builder-container'); // Container is dynamic, so we delegate to document

        // Add Column/Row
        $(document).on('click', '.pct-add-product', addProduct);
        $(document).on('click', '.pct-add-feature', addFeature);

        // Remove Buttons
        $(document).on('click', '.pct-remove-product', removeProduct);
        $(document).on('click', '.pct-remove-feature', removeFeature);

        // Save on submit
        $('form#post').on('submit', function () {
            saveTableData();
        });

        // Toggle Listeners (Boolean Buttons)
        $(document).on('click', '.pct-bool-btn', function (e) {
            e.preventDefault();
            const $btn = $(this);
            const $parent = $btn.closest('.pct-bool-toggle');
            const $input = $parent.find('.pct-feature-value-input');
            const newVal = $btn.attr('data-val'); // Explicit attribute read

            // Update UI
            $parent.find('.pct-bool-btn').removeClass('active');
            $btn.addClass('active');

            // Update Input
            $input.val(newVal).trigger('change');
        });

        // Change listener for inputs to trigger save
        $(document).on('change', '#pct-builder-container input, #pct-builder-container select', function () {
            saveTableData();
        });

        // Theme settings
        $('#pct-accent-color, #pct-highlight-color, #pct-theme').on('change', function () {
            saveTableData();
        });
    });

    function seedDefaultData() {
        // Generate IDs dynamically
        const ts = Date.now();

        // Default Products
        tableData.products = [
            { id: 'prod_1_' + ts, name: 'Basic', price: '$4.95', badge: '', highlighted: false, button_text: 'Sign Up', button_link: '#', features: {} },
            { id: 'prod_2_' + ts, name: 'Standard', price: '$14.95', badge: 'Best Deal', highlighted: true, button_text: 'Sign Up', button_link: '#', features: {} },
            { id: 'prod_3_' + ts, name: 'Premium', price: '$24.95', badge: '', highlighted: false, button_text: 'Sign Up', button_link: '#', features: {} }
        ];

        // Default Features
        const feat1 = 'feat_1_' + ts;
        const feat2 = 'feat_2_' + ts;
        const feat3 = 'feat_3_' + ts;

        tableData.features = [
            { id: feat1, name: 'Users', type: 'text' },
            { id: feat2, name: 'Storage', type: 'text' },
            { id: feat3, name: 'Support', type: 'bool' }
        ];

        // Populate Values
        tableData.products[0].features[feat1] = '1 User';
        tableData.products[0].features[feat2] = '50 GB';
        tableData.products[0].features[feat3] = 'false';

        tableData.products[1].features[feat1] = '3 Users';
        tableData.products[1].features[feat2] = '2000 GB';
        tableData.products[1].features[feat3] = 'true';

        tableData.products[2].features[feat1] = '10 Users';
        tableData.products[2].features[feat2] = 'Unlimited';
        tableData.products[2].features[feat3] = 'true';
    }

    function renderBuilder() {
        renderMatrix();
    }

    function renderMatrix() {
        const container = $('#pct-builder-container'); // Reusing existing container ID? 
        // Original HTML had #pct-products and #pct-features inside #pct-table-builder-meta
        // We should clear the meta box content and inject our table.
        // Wait, the meta box HTML is defined in PHP.
        // We need to target the wrapper.
        // Let's assume there is a wrapper or we create one.
        // In PHP, meta box has: <div id="pct-builder-wrapper">...</div>?
        // Let's use #pct-table-builder-meta .inside ? 
        // Or cleaner: Empty the existing containers and inject table.

        // Actually, let's look at `renderProducts` logic. It targeted `#pct-products`.
        // I will target `#pct-products` parent?
        // Let's clear `#pct-products` and `#pct-features` and hide them, then append matrix to a new container?
        // Better: Replace content of `#pct-table-builder-meta .inside`.

        // Finding the right container
        let $wrapper = $('#pct-builder-container');
        if ($wrapper.length === 0) {
            // Create if not exists (migrating structure)
            const $legacySection = $('#pct-products').closest('.pct-section');
            if ($legacySection.length) {
                $legacySection.before('<div id="pct-builder-container"></div>');
                // Hide Legacy Sections
                $legacySection.hide();
                $('#pct-features').closest('.pct-section').hide();
            } else {
                // Fallback if structure not found
                $('#pct-table-builder-meta .inside').prepend('<div id="pct-builder-container"></div>');
            }

            $wrapper = $('#pct-builder-container');
        }

        $wrapper.empty();

        // Check sidebar setting (default true)
        const showSidebar = (typeof tableData.settings.show_sidebar !== 'undefined') ? tableData.settings.show_sidebar : true;

        let html = '<table class="pct-matrix-table">';

        // -- THEAD (Products) --
        html += '<thead>';
        html += '<tr>';

        // Top-Left Corner (Action Bar / Legend)
        html += '<th><div class="pct-matrix-legend">';
        html += '<label style="font-size:12px;cursor:pointer;"><input type="checkbox" id="pct-toggle-sidebar" ' + (showSidebar ? 'checked' : '') + '> Show Row Headers</label>';
        html += '</div></th>';

        tableData.products.forEach((product, pIndex) => {
            html += `<th data-col="${pIndex}">
                <div class="pct-matrix-prop">
                    <button type="button" class="pct-remove-btn pct-remove-product" data-index="${pIndex}" style="float:right;">&times; Remove</button>
                    <span class="pct-matrix-prop-label">Name</span>
                    <input type="text" class="pct-product-name" value="${escapeHtml(product.name)}" data-index="${pIndex}">
                </div>
                  <div class="pct-matrix-prop" style="margin-top:5px;">
                    <span class="pct-matrix-prop-label">Price</span>
                    <input type="text" class="pct-product-price" value="${escapeHtml(product.price)}" data-index="${pIndex}">
                </div>
                 <div class="pct-matrix-prop" style="margin-top:5px;">
                    <span class="pct-matrix-prop-label">Badge</span>
                    <input type="text" class="pct-product-badge" value="${escapeHtml(product.badge)}" data-index="${pIndex}" placeholder="e.g. Best Value">
                </div>
                <div class="pct-matrix-prop" style="margin-top:5px;">
                    <label style="font-size:11px;">
                        <input type="checkbox" class="pct-product-highlighted" ${product.highlighted ? 'checked' : ''} data-index="${pIndex}"> Highlight
                    </label>
                </div>
                <!-- Button details could be collapsed or added here -->
                 <div class="pct-matrix-prop" style="margin-top:5px;">
                    <span class="pct-matrix-prop-label">Button Text</span>
                    <input type="text" class="pct-product-button-text" value="${escapeHtml(product.button_text)}" data-index="${pIndex}">
                </div>
                <div class="pct-matrix-prop" style="margin-top:5px;">
                    <span class="pct-matrix-prop-label">Button Link</span>
                    <input type="url" class="pct-product-button-link" value="${escapeHtml(product.button_link)}" data-index="${pIndex}">
                </div>
            </th>`;
        });
        html += '</tr></thead>';

        // -- TBODY (Features) --
        html += '<tbody>';
        tableData.features.forEach((feature, fIndex) => {
            // Default to text
            if (!feature.type) feature.type = 'text';

            // IF No Sidebar, force type text for rendering logic (but keep data intact if feasible, or just render inputs)
            // User said "if we choose no sidebar ... there is no check/cross".
            const forceText = !showSidebar;
            const effectiveType = forceText ? 'text' : feature.type;

            html += `<tr data-row="${fIndex}">`;

            // Feature Header Cell (Sticky Left)
            html += `<td>
                <div class="pct-feature-controls">
                    <span class="dashicons dashicons-menu pct-drag-handle"></span>
                    <button type="button" class="pct-remove-btn pct-remove-feature" data-index="${fIndex}">&times;</button>
                </div>
                
                <div class="pct-sidebar-inputs" style="${!showSidebar ? 'display:none;' : ''}">
                    <input type="text" class="pct-feature-name" value="${escapeHtml(feature.name)}" data-index="${fIndex}" placeholder="Feature Name" style="font-weight:bold; margin-bottom: 5px;">
                    <select class="pct-feature-type" data-index="${fIndex}" style="width:100%;">
                        <option value="text" ${feature.type === 'text' ? 'selected' : ''}>Text</option>
                        <option value="bool" ${feature.type === 'bool' ? 'selected' : ''}>Check / Cross</option>
                    </select>
                </div>
                 <div class="pct-sidebar-hidden-msg" style="${showSidebar ? 'display:none;' : 'font-size:11px;color:#aaa;font-style:italic;'}">
                    (Feature Row ${fIndex + 1})
                </div>
            </td>`;

            // Feature Value Cells (Per Product)
            tableData.products.forEach((product, pIndex) => {
                let value = (product.features && typeof product.features[feature.id] !== 'undefined') ? product.features[feature.id] : '';

                html += `<td>`;

                if (effectiveType === 'bool') {
                    const isTrue = (value === 'true' || value === true || value === '1');
                    html += `
                        <div class="pct-bool-toggle">
                            <button type="button" class="pct-bool-btn pct-bool-yes ${isTrue ? 'active' : ''}" data-val="true" title="Check">✓</button>
                            <button type="button" class="pct-bool-btn pct-bool-no ${!isTrue ? 'active' : ''}" data-val="false" title="Cross">✗</button>
                            <input type="hidden" class="pct-feature-value-input" data-product="${pIndex}" data-feature="${feature.id}" value="${escapeHtml(value)}">
                        </div>`;
                } else {
                    html += `<input type="text" class="pct-feature-value-input" value="${escapeHtml(value)}" data-product="${pIndex}" data-feature="${feature.id}">`;
                }

                html += `</td>`;
            });

            html += `</tr>`;
        });
        html += '</tbody>';

        html += '</table>';

        // Action Bar for Adding Cols/Rows
        html += `<div class="pct-action-bar">
            <button type="button" class="button button-primary pct-add-product">+ Add Plan (Column)</button>
            <button type="button" class="button button-secondary pct-add-feature">+ Add Feature (Row)</button>
        </div>`;

        $wrapper.html(html);
    }

    // -- CRUD Operations --

    function addProduct() {
        const newProduct = {
            id: 'prod_' + Date.now(),
            name: 'New Plan',
            price: '$0',
            badge: '',
            highlighted: false,
            button_text: 'Sign Up',
            button_link: '#',
            features: {}
        };
        tableData.products.push(newProduct);
        renderBuilder();
        saveTableData();
    }

    function removeProduct() {
        const index = $(this).data('index');
        if (confirm('Remove this plan column?')) {
            tableData.products.splice(index, 1);
            renderBuilder();
            saveTableData();
        }
    }

    function addFeature() {
        const newFeature = {
            id: 'feat_' + Date.now(),
            name: 'New Feature',
            type: 'text'
        };
        tableData.features.push(newFeature);
        renderBuilder();
        saveTableData();
    }

    function removeFeature() {
        const index = $(this).data('index');
        if (confirm('Remove this feature row?')) {
            const featureId = tableData.features[index].id;
            // Cleanup
            tableData.products.forEach(p => {
                if (p.features && p.features[featureId]) delete p.features[featureId];
            });
            tableData.features.splice(index, 1);
            renderBuilder();
            saveTableData();
        }
    }

    // Theme settings
    $('#pct-accent-color, #pct-highlight-color, #pct-theme').on('change', function () {
        saveTableData();
    });

    // Sidebar Toggle
    $(document).on('change', '#pct-toggle-sidebar', function () {
        // Update model directly because if we just save, we might lose other pending edits? 
        // Actually, safe to just save then render?
        // But saving relies on scraping the DOM. The checkbox IS in the DOM.
        // So saveTableData() will capture the new checkbox state.
        // Then we need to re-render to reflect the visual change (hide/show inputs).
        saveTableData();
        renderBuilder();
    });

    // Feature Type Change (Must re-render to switch input type)
    $(document).on('change', '.pct-feature-type', function () {
        saveTableData();
        renderBuilder();
    });

    // -- Scraping Logic --
    // Since we use delegation and explicit update on inputs, we can construct the data
    // slightly differently or just stick to scraping from DOM which is robust.
    // However, input events trigger 'saveTableData' which calls 'scrapeDataFromDOM'.
    // We MUST update scrapeDataFromDOM to handle matrix structure.

    function scrapeDataFromDOM() {
        const newData = {
            settings: {
                theme: $('#pct-theme').val(),
                accent_color: $('#pct-accent-color').val(),
                highlight_color: $('#pct-highlight-color').val(),
                show_sidebar: $('#pct-toggle-sidebar').is(':checked') // Capture Sidebar Setting
            },
            products: [],
            features: []
        };

        // 1. Scrape Products (Columns) from THEAD
        const $headerCells = $('.pct-matrix-table thead th').not(':first'); // Skip legend

        $headerCells.each(function (index) {
            const $th = $(this);
            const product = {
                id: tableData.products[index] ? tableData.products[index].id : 'prod_' + Date.now(), // Preserve ID if possible
                name: $th.find('.pct-product-name').val(),
                price: $th.find('.pct-product-price').val(),
                badge: $th.find('.pct-product-badge').val(),
                button_text: $th.find('.pct-product-button-text').val(),
                button_link: $th.find('.pct-product-button-link').val(),
                highlighted: $th.find('.pct-product-highlighted').is(':checked'),
                features: {}
            };
            newData.products.push(product);
        });

        // 2. Scrape Features (Rows) and Values
        const $rows = $('.pct-matrix-table tbody tr');
        $rows.each(function (fIndex) {
            const $tr = $(this);
            const feature = {
                id: tableData.features[fIndex] ? tableData.features[fIndex].id : 'feat_' + Date.now(),
                name: $tr.find('.pct-feature-name').val(),
                type: $tr.find('.pct-feature-type').val()
            };
            newData.features.push(feature);

            // Scrape Values in this row
            $tr.find('.pct-feature-value-input').each(function () {
                const pIndex = $(this).data('product');
                const val = $(this).val();

                if (newData.products[pIndex]) {
                    newData.products[pIndex].features[feature.id] = val;
                }
            });
        });

        return newData;
    }

    function saveTableData() {
        const freshData = scrapeDataFromDOM();
        tableData = freshData;
        const jsonString = JSON.stringify(freshData);
        const base64String = btoa(unescape(encodeURIComponent(jsonString)));
        $('#pct-table-data').val(base64String);
        // console.log('Matrix Saved', freshData);
    }

    function escapeHtml(text) {
        if (!text) return '';
        const map = { '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#039;' };
        return String(text).replace(/[&<>"']/g, function (m) { return map[m]; });
    }

})(jQuery);
